function all_bodies_labeled_uint8 = f4_generate_binary_from_coordinates(vertices_mat, edges_mat, faces_mat, bodies_mat, params)

% Description
% Function takes matrices with coordinate information obtained from Surface
% Evolver files. Function iterates over each body, iterates over each
% vertex in this body, maps [x,y,z] cartesian coordinates of this vertex to 
% pixel-coordinates of future image.
% Mapping is done as:
% - translate the vertex by the center of mass of all bodies
% - scale all dimension by the same number to about [-1, 1]
% - scale by Image size
% - translate to Image center
%
% Dmitry Ershov
% 2017 October

disp(' ')
disp(['Generating Binary Matrices from SE Coordinates...']);


%% 1. Find all coordinates [x,y,z] of all vertices in all bodies
all_faces_ids       = abs(bodies_mat(:,2));                                     % choose all faces ids from all bodies

faces_mat_abs       = abs(faces_mat);                                           % remove directional information (remove "-")
ind                 = ismember(faces_mat_abs(:, 1), all_faces_ids);             % ind of all faces
all_edges_ids       = faces_mat_abs(ind, 2:4);                                  % find all edges ids from all bodies    
ind                 = ismember(edges_mat(:, 1), all_edges_ids(:));              % ind of all edges
all_verts_ids       = edges_mat(ind, 2:3);                                      % find all verts ids from all bodies  
ind                 = ismember(vertices_mat(:,1), all_verts_ids(:));            % ind of all vertices
all_verts_coords    = vertices_mat(ind, 2:4);                                   % find all coords [x,y,z] of all verts


% Let's directly remap original coordinates to the Image coordinates.
actual_coords_span_xyz  = max(all_verts_coords,[],1) - min(all_verts_coords,[],1);  % span of all bodies in x,y,z
desired_image_span_XY   = [params.image_size(1), params.image_size(2)];         % x,y size of the image (x=width,cols and  y = height, rows)

% find the proper ratio in XY/xy systems
% min of XY spans divide by maximum of xy spans -  to be safe.
safety_margin           = 10;
R                       = floor(min(desired_image_span_XY       - 2*safety_margin) / max(actual_coords_span_xyz(1:2)));
desired_z               = ceil(R * actual_coords_span_xyz(3))   + 2*safety_margin; % 2*safety margin - because from 2 sides
params.image_size       = [desired_image_span_XY, desired_z];

% directly remap coordinates of vertices:
all_bodies_center                     = mean(all_verts_coords, 1);                                % center of mass of all bodies
all_verts_coords_centered             = all_verts_coords - repmat(all_bodies_center, size(all_verts_coords, 1), 1);  % translate all coordinates to "0" - subtract the center of mass of all bodies.
all_verts_coords_centered_scaled      = all_verts_coords_centered *  R;                           % this will rescale to the maximum span of [-0.5 , 0.5]
all_verts_coords_centered_scaled_tr   = all_verts_coords_centered_scaled + repmat(round(params.image_size / 2), size(all_verts_coords, 1), 1);        % confined the body to the indicated range of space.

% update the original vertices with these coordinates:
vertices_mat_corr           = vertices_mat(ind, :);
vertices_mat_corr(:, 2:4)      = all_verts_coords_centered_scaled_tr;

% figure;
% hold on
% plot3(all_verts_coords(:,1), all_verts_coords(:,2),all_verts_coords(:,3),'b.')
% plot3(all_verts_coords_centered(:,1), all_verts_coords_centered(:,2), all_verts_coords_centered(:,3),'r.')
% plot3(all_verts_coords_centered_scaled(:,1), all_verts_coords_centered_scaled(:,2), all_verts_coords_centered_scaled(:,3),'g.')
% plot3(all_verts_coords_centered_scaled_tr(:,1), all_verts_coords_centered_scaled_tr(:,2), all_verts_coords_centered_scaled_tr(:,3),'m.')
% legend('original', 'centered', 'centered-scaled', 'c-s-translated')




%% 2. Make the 3d uint8 matrix representing each body
% Values in the uint8 matrix will be:
% 0 = interior of a body
% 1 = exterior of a body.

all_bodies_Ns           = unique(bodies_mat(:, 1))';                                % number of bodies
all_bodies_uint8_cell   = cell(1, length(all_bodies_Ns));                         % Cell holder of body matrices;

zero_mat_uint8          = uint8(zeros(params.image_size(2),...
                                      params.image_size(1),...
                                      params.image_size(3)));                       % 8bit matrix of 0, a template. [x,y,z] = [2,1,3] dimensions of mat.
                              


% area of a face should not exceed area of a triangle 
% with each side < 1; say side = 0.5; then A = (a+b+c)/2
% l = 0.5;
% area_upper_limit = (0.5 + 0.5 + 0.5)/2 ;
        
for  body_ind = all_bodies_Ns
    
    disp(['    Calculating body ' num2str(body_ind) ' out of ' num2str(length(all_bodies_Ns))])
    
    % find current body all vertices and their [x,y,z] coordinates 
    body_i            = all_bodies_Ns(body_ind);   
    body_i_faces      = abs(bodies_mat(bodies_mat(:,1) == body_i, 2)); 
    body_i_faces_ind  = ismember(faces_mat(:,1), body_i_faces);
    body_i_edges      = unique(abs(faces_mat(body_i_faces_ind, 2:4)));
    body_i_vert_ind   = ismember(edges_mat(:,1), body_i_edges);
    body_i_verts      = edges_mat(body_i_vert_ind, 2:3);
    body_i_coords_ind = ismember(vertices_mat_corr(:,1), body_i_verts);
    body_i_coords     = vertices_mat_corr(body_i_coords_ind, 2:4); 
    body_i_coords_final  = body_i_coords;

    
    % one can check the results of body selection
    if 0 
        figure; 
        hold on;
        axis equal; 
        xlabel('x'); ylabel('y'); zlabel('z')
        xlim([1 params.image_size(1)]); ylim([1 params.image_size(2)]); zlim([1 params.image_size(3)])
        scatter3(body_i_coords_final(:,1),...
                 body_i_coords_final(:,2),...
                 body_i_coords_final(:,3), 'r.')
    end

    
    
    % one of making the 3D discrete matrix: via reconstructing from faces.
    % Advantage: each face can be subdivided as many times as needed.
    % this helps to fill the gaps between coordinates.
    body_i_mat_uint8_cortex = zero_mat_uint8;
    N_faces = length(body_i_faces);
    
    for ii = 1 : N_faces
        
        fi              = body_i_faces(ii);
        fi_ind          = ismember(faces_mat(:,1), fi);
        fi_edges        = unique(abs(faces_mat(fi_ind, 2:4)));
        fi_vert_ind     = ismember(edges_mat(:,1), fi_edges');
        fi_verts        = edges_mat(fi_vert_ind, 2:3);
        fi_coords_ind   = ismember(vertices_mat_corr(:,1), unique(fi_verts));
        fi_coords       = vertices_mat_corr(fi_coords_ind, 2:4);  
        
        v1              = fi_coords;                        % coordinates of this face    
        f1              = [1, 2, 3];                        % fake vertices indices for this face
        
        max_l       = max([sqrt(sum((v1(2, :) - v1(1, :)).^2)), ...         % side lengths:
                           sqrt(sum((v1(3, :) - v1(1, :)).^2)),...
                           sqrt(sum((v1(3, :) - v1(2, :)).^2))]);
    
        while max_l  > 0.5 
            [v1, f1]        = subdivide_tri(v1, f1);
            % new vertices v1, new faces f1
            for zz = 1 : size(f1, 1)
                v2      = v1(f1(zz, :)', :);
                this_l   = max([sqrt(sum((v2(2, :) - v2(1, :)).^2)), ...         % side lengths:
                           sqrt(sum((v2(3, :) - v2(1, :)).^2)),...
                           sqrt(sum((v2(3, :) - v2(2, :)).^2))]);

                max_l = min([max_l, this_l]);
            end
        end   

        for n = 1 : size(v1, 1)
            body_i_mat_uint8_cortex(floor(v1(n, 2)), floor(v1(n, 1)), floor(v1(n, 3))) = 1;
        end  
        
        if mod(ii, 100) == 0
            disp(['    finished faces: ' num2str(ii) ' out of ' num2str(N_faces)])
        end
    end
    
        
    % convert coordinates into discrete matrix uint8:
    % It will have only the "cortex", because the coordinates describe only
    % outline of the shape.
%     body_i_mat_uint8_cortex = zero_mat_uint8;
%     for n = 1 : size(body_i_coords_final, 1)
%         body_i_mat_uint8_cortex(floor(body_i_coords_final(n, 2)),...
%                          floor(body_i_coords_final(n, 1)),...
%                          floor(body_i_coords_final(n, 3))) = 1;
%     end   

    
    % fill in the cortex:
    body_i_mat_uint8_full = body_i_mat_uint8_cortex;
    
    for n = 1 : size(body_i_mat_uint8_cortex, 3)  
        
        % there might be discontinuities in the cortex; we close them first:
        body_i_mat_uint8_cortex_closed = imclose(body_i_mat_uint8_cortex(:,:,n), strel('disk', params.image_open_cortex_by));     
        
        % then we fill the fied cortex:
        body_i_mat_uint8_full(:,:,n) = bwfill(body_i_mat_uint8_cortex_closed, 'holes'); 
        
        % remove the cortex:  take only interior. This is to prevent bodies
        % touching each other.
        body_i_mat_uint8_full(:,:,n) = body_i_mat_uint8_full(:,:,n) - body_i_mat_uint8_cortex_closed;
    end
    
    % by convetion, interior is black, exterior is white; for this invert
    % them. Keep it uint8.
    body_i_mat_uint8_full     = uint8(~body_i_mat_uint8_full);

    % collect all bodies into the cell
    all_bodies_uint8_cell{body_ind} = body_i_mat_uint8_full;     

        
    % DEBUG: the current matrix may contain imperfections
    % one can see the 3D matrix:
    if 0
        figure;    
        imagesc(body_i_mat_uint8_full(:,:,1))
        hold on;
        ax = gca;
        ch = get(ax,'Children');
        for i = 80 : 140% size(body_i_mat_uint8_filled, 3)
            set(ch, 'CData', body_i_mat_uint8_full(:, :, i));
            title(['slice ' num2str(i)])
            pause(10/100)
        end
    end

end



% Put all found separate bodies onto one; label them according to the 
% body index. (analogous to bwlabel)
% inversion here is needed only to conform to the convention
all_bodies_labeled_uint8    = uint8(zeros(size(all_bodies_uint8_cell{1})));
for i = 1 : length(all_bodies_uint8_cell)
    all_bodies_labeled_uint8(~logical(all_bodies_uint8_cell{i})) = i;
end
% all_bodies_uint8 = uint8(~all_bodies_uint8);


disp(['    Done']);


% One can implement 3D rotation for coordinates, if needed.
% Here are some ideas:
%     % Center to (0,0,0) and apply rotation matrices.   
%     % 3D rotation first in ZY plane and second in ZX plane:
%     % rotation in (Z,Y) - plane.
%     angle_rot_zy    = 86; 
%     % rotation in (Z,X) - plane.
%     angle_rot_zx    = 2.5; 
%     
%     xxx = all_verts_coords(:,1)';
%     yyy = all_verts_coords(:,2)';
%     zzz = all_verts_coords(:,3)';    
%    
%     % first rotation
%     R_zy            = [cosd(angle_rot_zy) -sind(angle_rot_zy);...
%                        sind(angle_rot_zy) cosd(angle_rot_zy)];
%     coordRot        = R_zy * [yyy; zzz];      
%     % new coords:
%     xxx2            = xxx;
%     yyy2            = coordRot(1,:);
%     zzz2            = coordRot(2,:);
%     
%     % second rotation
%     R_zx            = [cosd(angle_rot_zx) -sind(angle_rot_zx);...
%                        sind(angle_rot_zx) cosd(angle_rot_zx)];
%     coordRot        = R_zx * [xxx2; zzz2];      
%     xxx3            = coordRot(1,:);
%     yyy3            = yyy2;
%     zzz3            = coordRot(2,:);
%         
%     all_verts_coords_rot = [xxx3(:), yyy3(:), zzz3(:)];



